﻿Imports System
Imports System.Drawing
Imports System.Collections
Imports System.IO
Imports System.Runtime.Serialization.Formatters.Binary

' Note that this class requires a reference to System.Drawing.dll.

' All classes whose instances can participate as characters in
' the game must implement this interface
Public Interface ICharacter
    Inherits ICloneable
    Property Name() As String
    Property Image() As Image
    Property Strength() As Integer
    ' ...
End Interface 'ICharacter

' An abstract class for implementing game characters
Public MustInherit Class Character
    Implements ICharacter
    Private myName As String
    Private myImage As Image
    Private myStrength As Integer

    ' The character's name
    Public Property Name() As String Implements ICharacter.Name
        Get
            Return myName
        End Get
        Set(ByVal Value As String)
            myName = Value
        End Set
    End Property

    ' An image of the character
    Public Property Image() As Image Implements ICharacter.Image
        Get
            Return myImage
        End Get
        Set(ByVal Value As Image)
            myImage = Value
        End Set
    End Property

    ' The character's physical strength
    Public Property Strength() As Integer _
                    Implements ICharacter.Strength
        Get
            Return myStrength
        End Get
        Set(ByVal Value As Integer)
            myStrength = Value
        End Set
    End Property

    ' Return a copy of this Character object.
    Public Function Clone() As Object _
                    Implements ICloneable.Clone
        Return Me.MemberwiseClone()
    End Function

    ' ...
End Class 'Character

' A class for representing hero characters
Public Class Hero
    Inherits Character
    Private myBravery As Integer

    Public Property Bravery() As Integer
        Get
            Return myBravery
        End Get
        Set(ByVal Value As Integer)
            myBravery = Value
        End Set
    End Property

End Class 'Hero

' An instance of this class keeps a collection of prototypical
' character object and creates new character objects by copying
' the prototypical objects.
Public Class CharacterManager
    Private characters As New ArrayList()
    Private Shared rand As New Random()

    '
    ' Get a copy of random character from the collection.
    '
    ReadOnly Property RandomCharacter() As Character
        Get
            Dim i As Integer = rand.Next(characters.Count)
            Dim c As Character = CType(characters(i), Character)
            Return CType(c.Clone(), Character)
        End Get
    End Property

    '
    ' Add a prototypical object to the collection.
    '
    Public Sub AddCharacter(ByVal character As Character)
        characters.Add(character)
    End Sub

End Class 'CharacterManager

' This class loads character objects and adds them to the
' the CharacterManager.
'
Class CharacterLoader
    Private mgr As CharacterManager

    'cm - The CharacterManager this object will work with.
    Sub New(ByVal cm As CharacterManager)
        mgr = cm
    End Sub

    ' Load character objects from the specified file.
    ' Since failure only affects the rest of the program to
    ' the extent that new character objects are not loaded,
    ' we need not throw any exceptions.
    Function loadCharacters(ByVal fname As String) As Integer
        Dim formatter As New BinaryFormatter()
        Dim reader As Stream = File.OpenRead(fname)
        Dim objectCount As Integer = 0 ' Number of objects loaded
        ' If construction of InputStream fails, just return
        Try
            While True
                Dim o As Object = formatter.Deserialize(reader)
                If TypeOf o Is Character Then
                    mgr.AddCharacter(CType(o, Character))
                    objectCount += 1
                End If
            End While
            ' try
        Finally
            If reader IsNot Nothing Then
                reader.Close()
            End If
        End Try
        Return objectCount
    End Function 'loadCharacters

End Class 'CharacterLoader
